import json, os, subprocess, sys
from datetime import datetime, timezone

ROOT = os.path.abspath(os.path.join(os.path.dirname(__file__), ".."))
ENGINE = os.path.join(ROOT, "engine_h22_transitions_lines.py")
MANIFEST = os.path.join(ROOT, "configs", "manifest_h22.json")
RUNS = os.path.join(ROOT, "runs")

def utc_ts():
    return datetime.now(timezone.utc).strftime("%Y-%m-%dT%H-%M-%SZ")

def main():
    if not os.path.exists(RUNS):
        os.makedirs(RUNS, exist_ok=True)
    outdir = os.path.join(RUNS, utc_ts())
    # Pre-create structure; engine will also ensure subdirs
    for sd in ["config", "outputs/metrics", "outputs/audits", "outputs/run_info", "logs"]:
        os.makedirs(os.path.join(outdir, sd), exist_ok=True)

    cmd = [sys.executable, ENGINE, "--manifest", MANIFEST, "--outdir", outdir]
    print("Running:", " ".join(cmd))
    proc = subprocess.run(cmd, capture_output=True, text=True, cwd=ROOT)
    # Store stdout/stderr
    with open(os.path.join(outdir, "logs", "run_stdout.txt"), "w", encoding="utf-8") as f:
        f.write(proc.stdout)
    with open(os.path.join(outdir, "logs", "run_stderr.txt"), "w", encoding="utf-8") as f:
        f.write(proc.stderr)

    # Read audit for pass/fail
    audit_path = os.path.join(outdir, "outputs", "audits", "h22_audit.json")
    if not os.path.exists(audit_path):
        print("ERROR: audit not produced:", audit_path)
        sys.exit(2)
    with open(audit_path, "r", encoding="utf-8") as f:
        aud = json.load(f)

    # Print concise summary
    print("Result: PASS=" + str(aud.get("pass")))
    print("P1=", aud.get("P1"), "P2=", aud.get("P2"))
    on = aud.get("on", {})
    ref = aud.get("ref", {})
    print("ON:  A1={:.3f}  A2={:.3f}  S1={:.2f}  S2={:.2f}  R2={:.3f}".format(on.get("A1", float("nan")), on.get("A2", float("nan")), on.get("S1", float("nan")), on.get("S2", float("nan")), on.get("R2", float("nan"))))
    print("REF: A1={:.3f}  A2={:.3f}".format(ref.get("A1", float("nan")), ref.get("A2", float("nan"))))

    # Verify PASS
    if not aud.get("pass"):
        print("FAIL: acceptance gates not met. See audit JSON and periodograms under:", outdir)
        sys.exit(1)

    # Also surface the result_line
    res_line_path = os.path.join(outdir, "outputs", "run_info", "result_line.txt")
    if os.path.exists(res_line_path):
        with open(res_line_path, "r", encoding="utf-8") as f:
            print("Result line:", f.read().strip())

    print("Outputs in:", outdir)

if __name__ == "__main__":
    main()
